#include <Servo.h>

// ====== Configuración de inversión por eje ======
const bool invertBase     = true;   // Izquierda mueve a la IZQUIERDA (invierte sentido original)
const bool invertShoulder = false;   // Empujar arriba SUBE (invierte sentido original)
const bool invertElbow    = true;  // Cambiá si fuese necesario
const bool invertGripper  = false;  // Cambiá si fuese necesario

// ====== Hardware ======
Servo servoBase;
Servo servoShoulder;
Servo servoElbow;
Servo servoGripper;

const int PinTestigo   = 3;   // LED testigo (digital 3)
const int basePin      = 11;  //  Base
const int shoulderPin  = 9;   //  SERVO L
const int elbowPin     = 10;  //  SERVO R
const int gripperPin   = 6;   //  PINZA

// Joystick en entradas analógicas (A0..A3)
const int xdirPin = 0;   // A0 → Base (izq/der)
const int ydirPin = 1;   // A1 → Shoulder (arriba/abajo)
const int zdirPin = 2;   // A2 → Elbow
const int gdirPin = 3;   // A3 → Gripper

// ====== Parámetros de control ======
const int movementDelay = 10;     // ms entre pasos
const int stepPerTick   = 1;      // grados por “tick”
const int deadzone      = 80;     // zona muerta alrededor de 512

// Límites personalizados
const int minBaseAngle      = 0;
const int maxBaseAngle      = 170;
const int minShoulderAngle  = 115;
const int maxShoulderAngle  = 180;
const int minElbowAngle     = 0; 
const int maxElbowAngle     = 130;
const int minGripperAngle   = 130;
const int maxGripperAngle   = 180;

// ====== Funciones ======
int getDirectionFromJoystick(int analogValue) {
  // Mapea a -1, 0 o +1 con zona muerta centrada en 512
  if (analogValue < (512 - deadzone)) return -1; // hacia “abajo/izquierda”
  if (analogValue > (512 + deadzone)) return +1; // hacia “arriba/derecha”
  return 0;
}

void moveServoWithDirection(Servo &servo, int direction, int minAngle, int maxAngle, bool invertAxis) {
  int currentAngle = servo.read();
  int delta = (invertAxis ? -direction : direction) * stepPerTick;
  int newAngle = constrain(currentAngle + delta, minAngle, maxAngle);
  if (newAngle != currentAngle) servo.write(newAngle);
}

void setup() {
  pinMode(PinTestigo, OUTPUT);
  digitalWrite(PinTestigo, HIGH);

  servoBase.attach(basePin);
  servoShoulder.attach(shoulderPin);
  servoElbow.attach(elbowPin);
  servoGripper.attach(gripperPin);

  // Opcional: llevar a posiciones iniciales seguras
  servoBase.write((minBaseAngle + maxBaseAngle) / 2);
  servoShoulder.write(minShoulderAngle);
  servoElbow.write(minElbowAngle);
  servoGripper.write(minGripperAngle);
}

void loop() {
  int xDir = analogRead(xdirPin); // Base
  int yDir = analogRead(ydirPin); // Hombro
  int zDir = analogRead(zdirPin); // Codo
  int gDir = analogRead(gdirPin); // Pinza

  int baseDirection     = getDirectionFromJoystick(xDir);
  int shoulderDirection = getDirectionFromJoystick(yDir);
  int elbowDirection    = getDirectionFromJoystick(zDir);
  int gripperDirection  = getDirectionFromJoystick(gDir);

  moveServoWithDirection(servoBase,     baseDirection,     minBaseAngle,     maxBaseAngle,     invertBase);
  moveServoWithDirection(servoShoulder, shoulderDirection, minShoulderAngle, maxShoulderAngle, invertShoulder);
  moveServoWithDirection(servoElbow,    elbowDirection,    minElbowAngle,    maxElbowAngle,    invertElbow);
  moveServoWithDirection(servoGripper,  gripperDirection,  minGripperAngle,  maxGripperAngle,  invertGripper);

  delay(movementDelay);
}
